<?php
/**
 * Core: OxyProps Legaacy Settings
 *
 * This file handles legacy options and settings and manages transition from previous
 * versions to the latest one.
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  Core
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.4.0 Creation.
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * OxyProps\Inc\Legacy.
 *
 * The Legacy Class.
 *
 * @since 1.4.0 Creation.
 * @since 1.5.0 Convert to Singleton Trait and get rid of Base Controller.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 */
class Legacy extends Base_Controller {

	use Singleton;

	/**
	 * Legacy options to latest options mapping.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @var array $mapping Key is old option, Value is new option.
	 */
	private array $mapping;

	/**
	 * Initialize Legacy class.
	 *
	 * Called by the constructor. It retreives and populates all the class properties.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function register(): void {
		// If no legacy options exists in database, do nothing.
		if ( ! get_option( 'oxyprops_canvas_color' ) ) {
			return;
		}

		$this->mapping = array(
			'oxyprops_wants_context'                 => '',
			'oxyprops_wants_elements'                => '',
			'oxyprops_wants_O2_reset'                => '',
			'oxyprops_wants_colors'                  => '',
			'oxyprops_wants_normalize'               => '',
			'oxyprops_wants_user_schemes_stylesheet' => '',
			'oxyprops_wants_colors_utilities'        => '',
			'oxyprops_wants_gradients_utilities'     => '',
			'oxyprops_wants_color_schemes_utilities' => '',
			'oxyprops_wants_shadows_utilities'       => '',
			'oxyprops_wants_aspect_ratios_utilities' => '',
			'oxyprops_wants_typography_utilities'    => '',
			'oxyprops_wants_easing_utilities'        => '',
			'oxyprops_wants_animations_utilities'    => '',
			'oxyprops_wants_sizes_utilities'         => '',
			'oxyprops_wants_borders_utilities'       => '',
			'oxyprops_wants_zindex_utilities'        => '',
			'oxyprops_wants_buttons_utilities'       => '',
			'oxyprops_wants_cards_utilities'         => '',
			'oxyprops_wants_layouts_utilities'       => '',
			'oxyprops_wants_grid12_utilities'        => '',
			'oxyprops_wants_all_utilities'           => '',
			'oxyprops_canvas_color'                  => '',
			'oxyprops_brand_color'                   => '',
			'oxyprops_links_color'                   => '',
			'oxyprops_visited_color'                 => '',
			'OxyProps_lic_email'                     => '',
			'oxyprops_ct_global_settings_bak'        => '',
			'oxyprops_user_custom_canvas'            => '',
			'oxyprops_user_custom_brand'             => '',
			'oxyprops_user_custom_links'             => '',
			'oxyprops_user_custom_visited'           => '',
			'OxyProps_license_Key'                   => '',
			'oxyprops_accent_color'                  => '',
			'oxyprops_user_custom_accent'            => '',
			'oxyprops_font_size_base'                => '',
			'oxyprops_scale_ratio'                   => '',
			'oxyprops_mobile_ratio'                  => '',
			'oxyprops_subheading_ratio'              => '',
			'oxyprops_font_size_stop'                => '',
			'oxyprops_user_custom_canvas_hex'        => '',
			'oxyprops_user_custom_brand_hex'         => '',
			'oxyprops_user_custom_accent_hex'        => '',
			'oxyprops_user_custom_links_hex'         => '',
			'oxyprops_user_custom_visited_hex'       => '',
			'OxyProps_lic_Key'                       => '',
		);

		$this->transfer_license();
		$this->transfer_options();
		$this->oxyprops_delete_deprecated();
	}

	/**
	 * Get new to legacy mapping.
	 *
	 * The returned array contains the new structure and their legacy equivaalents.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array Key is new, Value is legacy.
	 */
	public function get_oxyprops_legacy_mapping(): array {
		$legacy = array();

		$legacy['features'] = array(
			'context'           => 'oxyprops_wants_context',
			'elements'          => 'oxyprops_wants_elements',
			'o2_reset'          => 'oxyprops_wants_O2_reset',
			'global_colors'     => 'oxyprops_wants_colors',
			'normalize'         => 'oxyprops_wants_normalize',
			'stylesheet'        => 'oxyprops_wants_user_schemes_stylesheet',
			'dequeue_gutenberg' => false,
			'css_print'         => false,
		);

		$legacy['utilities'] = array(
			'all'           => 'oxyprops_wants_all_utilities',
			'colors'        => 'oxyprops_wants_colors_utilities',
			'gradients'     => 'oxyprops_wants_gradients_utilities',
			'color_schemes' => 'oxyprops_wants_color_schemes_utilities',
			'shadows'       => 'oxyprops_wants_shadows_utilities',
			'aspect_ratios' => 'oxyprops_wants_aspect_ratios_utilities',
			'typography'    => 'oxyprops_wants_typography_utilities',
			'easing'        => 'oxyprops_wants_easing_utilities',
			'animations'    => 'oxyprops_wants_animations_utilities',
			'sizes'         => 'oxyprops_wants_sizes_utilities',
			'borders'       => 'oxyprops_wants_borders_utilities',
			'zindex'        => 'oxyprops_wants_zindex_utilities',
			'buttons'       => 'oxyprops_wants_buttons_utilities',
			'cards'         => 'oxyprops_wants_cards_utilities',
			'layouts'       => 'oxyprops_wants_layouts_utilities',
			'grid12'        => 'oxyprops_wants_grid12_utilities',
		);

		$legacy['colors'] = array(
			'canvas'  => array(
				'name'   => 'oxyprops_canvas_color',
				'hex'    => 'oxyprops_user_custom_canvas_hex',
				'weight' => '',
				'swatch' => '',
				'load'   => '',
			),
			'brand'   => array(
				'name'   => 'oxyprops_brand_color',
				'hex'    => 'oxyprops_user_custom_brand_hex',
				'weight' => '',
				'swatch' => '',
				'load'   => '',
			),
			'accent'  => array(
				'name'   => 'oxyprops_accent_color',
				'hex'    => 'oxyprops_user_custom_accent_hex',
				'weight' => '',
				'swatch' => '',
				'load'   => '',
			),
			'links'   => array(
				'name'   => 'oxyprops_links_color',
				'hex'    => 'oxyprops_user_custom_links_hex',
				'weight' => '',
				'swatch' => '',
				'load'   => '',
			),
			'visited' => array(
				'name'   => 'oxyprops_visited_color',
				'hex'    => 'oxyprops_user_custom_visited_hex',
				'weight' => '',
				'swatch' => '',
				'load'   => '',
			),
		);

		$legacy['typography'] = array(
			'base_size'        => 'oxyprops_font_size_base',
			'minimum_size'     => 'oxyprops_font_size_stop',
			'scale_ratio'      => 'oxyprops_scale_ratio',
			'mobile_ratio'     => 'oxyprops_mobile_ratio',
			'subheading_ratio' => 'oxyprops_subheading_ratio',
		);

			return $legacy;
	}

	/**
	 * Lists the Plugin deprecated Settings.
	 *
	 * These options are deprecated and can simply be deleted from the database
	 * on plugin update.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return array List of deprecated options.
	 */
	public function get_oxyprops_deprecated_options(): array {
		$deprecated = array(
			'oxyprops_user_custom_canvas',
			'oxyprops_user_custom_brand',
			'oxyprops_user_custom_links',
			'oxyprops_user_custom_visited',
			'oxyprops_user_custom_accent',
		);

			return $deprecated;
	}

	/**
	 * Transfers legacy license option to latest version.
	 *
	 * Checks if license options exist, and if so, transferts information to the latest option.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function transfer_license(): void {
		$existing_email       = get_option( 'OxyProps_lic_email' );
		$existing_license     = get_option( 'OxyProps_lic_Key' );
		$existing_license_alt = get_option( 'OxyProps_license_Key' );
		$existing_oxy_backup  = get_option( 'oxyprops_ct_global_settings_bak' );
		$current_settings     = get_option( 'oxyprops', array() );
		$new_option           = $current_settings;
		$new_settings         = array();

		if ( false !== $existing_email && ! isset( $current_settings['license']['email'] ) ) {
			$new_settings['license']['email'] = $existing_email;
		}
		if ( false !== $existing_license && ! isset( $current_settings['license']['key'] ) ) {
			$new_settings['license']['key'] = $existing_license;
		}
		if ( false !== $existing_license_alt && ! isset( $new_settings['license']['key'] ) ) {
			$new_settings['license']['key'] = $existing_license_alt;
		}
		if ( false !== $existing_oxy_backup && ! isset( $current_settings['oxy_backup'] ) ) {
			$new_settings['oxy_backup'] = $existing_oxy_backup;
		}

		$response = update_option( 'oxyprops', array_replace_recursive( $new_option, $new_settings ) );

		if ( true !== $response ) {
			return;
		}

		$old_license      = $existing_license;
		$current_settings = get_option( 'oxyprops', array() );
		$new_license      = isset( $current_settings['license']['key'] ) ? $current_settings['license']['key'] : '';

		if ( $old_license !== $new_license ) {
			return;
		}

		// The transfer was OK, let's clean after us.
		delete_option( 'OxyProps_lic_email' );
		delete_option( 'OxyProps_lic_Key' );
		delete_option( 'OxyProps_license_Key' );
		delete_option( 'oxyprops_ct_global_settings_bak' );
	}

	/**
	 * Transfers options from legacy version to latest.
	 *
	 * @since 1.4.0 Creation.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function transfer_options(): void {
		$defaults = Init::get_instance( 'defaults' );
		if ( method_exists( $defaults, 'get_oxyprops_default_settings' ) ) {
			$default_settings = $defaults->get_oxyprops_default_settings();
		}
		$options_mapping  = $this->get_oxyprops_legacy_mapping();
		$current_settings = isset( get_option( 'oxyprops' )['settings'] ) ? get_option( 'oxyprops' )['settings'] : array();
		$new_settings     = array();
		$merged_settings  = array();

		foreach ( $options_mapping['features'] as $new => $legacy ) {
			$existing_option = get_option( $legacy, 'none' );
			if ( 'none' !== $existing_option && ! isset( $current_settings['features'][ $new ] ) ) {
				$new_settings['features'][ $new ] = $existing_option;
			}
		}
		foreach ( $options_mapping['utilities'] as $new => $legacy ) {
			$existing_option = get_option( $legacy, 'none' );
			if ( 'none' !== $existing_option && ! isset( $current_settings['utilities'][ $new ] ) ) {
				$new_settings['utilities'][ $new ] = $existing_option;
			}
		}
		foreach ( $options_mapping['colors'] as $color => $array ) {
			foreach ( $array as $new => $legacy ) {
				switch ( $new ) {
					case 'load':
						if ( ! isset( $current_settings['colors'][ $color ]['load'] ) ) {
							$new_settings['colors'][ $color ]['load'] = true;
						}
						break;
					case 'hex':
						$existing_option = get_option( $legacy );
						if ( $existing_option && ! isset( $current_settings['colors'][ $color ][ $new ] ) ) {
							$new_settings['colors'][ $color ][ $new ] = 'custom' !== $new_settings['colors'][ $color ]['name'] ? $default_settings['colors'][ $new_settings['colors'][ $color ]['name'] ]['hex'] : $existing_option;
						}
						break;
					case 'weight':
						if ( ! isset( $current_settings['colors'][ $color ]['weight'] ) ) {
							$new_settings['colors'][ $color ]['weight'] = '';
						}
						break;
					case 'name':
						$existing_option = get_option( $legacy );
						if ( $existing_option && ! isset( $current_settings['colors'][ $color ][ $new ] ) ) {
							$new_settings['colors'][ $color ][ $new ] = $existing_option;
						}
						break;
					case 'swatch':
						if ( ! isset( $current_settings['colors'][ $color ]['swatch'] ) ) {
							$new_settings['colors'][ $color ]['swatch'] = 'custom' !== $new_settings['colors'][ $color ]['name'] ? $default_settings['colors'][ $new_settings['colors'][ $color ]['name'] ]['swatch'] : array();
						}
						break;
				}
			}
		}
		foreach ( $options_mapping['typography'] as $new => $legacy ) {
			$existing_option = get_option( $legacy );
			if ( $existing_option && ! isset( $current_settings['typography'][ $new ] ) ) {
				$new_settings['typography'][ $new ] = $existing_option / 10;
			}
		}
		$merged_settings  = array_replace_recursive( $default_settings, $current_settings );
		$merged_settings  = array_replace_recursive( $merged_settings, $new_settings );
		$temp             = array();
		$temp['settings'] = $merged_settings;
		update_option( 'oxyprops', array_replace_recursive( get_option( 'oxyprops' ), $temp ) );
	}

	/**
	 * Deletes deprecated options.
	 *
	 * @since 1.4.0 Describe changes.
	 * @author Cédric Bontems <cedric@thewebforge.dev>
	 *
	 * @return void
	 */
	public function oxyprops_delete_deprecated(): void {
		$deprecated = $this->get_oxyprops_deprecated_options();
		foreach ( $deprecated as $option ) {
			delete_option( $option );
		}
	}
}


